package gov.va.med.domain.service.messaging;

import gov.va.med.domain.service.messaging.parse.HL7MessageMetadata;

import gov.va.med.mhv.rxrefill.data.repository.MessageTraceLogRepository;
import gov.va.med.mhv.rxrefill.data.model.MessageTraceLog;
import gov.va.med.mhv.rxrefill.exception.MHVRuntimeException;


import java.util.Date;
import java.util.List;
import java.io.PrintWriter;
import java.io.StringWriter;

import org.apache.commons.logging.Log;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

/**
 * Utility class to log messaging errors and messages flowing through the system.
 *
 * @author Slava Uchitel
 * @version $Id: MessagingLogHelper.java,v 1.7 2005/08/26 15:49:36 alex.kalinovsky Exp $
 * @since MHV 2.0 <br>
 *        Jul 5, 2005
 */

@Component
public class MessagingLogHelper {

	@Autowired
	private MessageTraceLogRepository messageTraceLogRepository;

   public static String getStackTrace(Throwable t) {
      String stackTrace = null;
      if (t.getStackTrace() != null) {
         StringWriter writer = new StringWriter();
         t.printStackTrace(new PrintWriter(writer));
         stackTrace = writer.getBuffer().toString();
      }
      return stackTrace;
   }

   //TODO: revisit the ERROR_QUE requirements
   /*private static JmsSender getJmsSender() {
      return (JmsSender) ObjectFactory.getBean(MessagingConstants.JMS_SENDER_BEAN_NAME);
   }*/

   public static void logMessagingError(Log logger, String loggerMessage, String message, Throwable t) {
      logMessagingError(logger, loggerMessage, message, t, null);
   }

   public static void logMessagingError(String message, Throwable t) {
      logMessagingError(null, null, message, t, null);
   }

   public static void logMessagingError(Log logger, String loggerMessage, String message, Throwable t, HL7MessageMetadata metaData) {
      if (logger != null) {
         // Log first to make sure we always know what exception has caused this error
         logger.error(loggerMessage, t);
      }

      String errorMessage = null;
      String stackTrace = null;
      if (t != null) {
         errorMessage = t.getMessage();
         stackTrace = getStackTrace(t);
      }
      else if (metaData != null && !metaData.isOk()) {
         errorMessage = metaData.getAckError();
         stackTrace = metaData.getAckText();
      }
      if (logger != null) {
         logger.debug("About to send error to error queue.\n\r" + errorMessage + "\n\r" + stackTrace + "\n\r" + message + "\n\r" + metaData);
      }
      //TODO: revisit the ERROR_QUE requirements
      //getJmsSender().sendObjectMessage(JmsSender.ERROR_QUEUE, message, stackTrace, errorMessage, false);
   }

   /**
    * Function for logging messagees flowing in trought the framework
    * <P>
    * Some scenarios use HL7 Message Type instead of Function name. The consideration here is that
    * this class is a general class which is not limited to HL7 messages. Therefore Function Name is
    * more generally applicable as the framework relies on it for all protocols.
    * 
    * Transaction method that requires a new transaction so that the logging (audit) of the message
    * is ensured if the main business flow transaction should error and roll back.
    *
    * @param userIdentifier normally a userid or icn but anything will work
    * @param message the actual message
    * @param messageType the message type or function under which this is logged
    */
   @Transactional(propagation = Propagation.REQUIRES_NEW)
   public void logInboundMessage(String userIdentifier, String messageType, String message) {

      logMessage(userIdentifier, messageType, message, "in");
   }

   /**
    * Function for logging messagees flowing out trought the framework
    * <P>
    * Some scenarios use HL7 Message Type instead of Function name. The consideration here is that
    * this class is a general class which is not limited to HL7 messages. Therefore Function Name is
    * more generally applicable as the framework relies on it for all protocols.
    * 
    * Transaction method that requires a new transaction so that the logging (audit) of the message
    * is ensured if the main business flow transaction should error and roll back.
    *
    * @param userIdentifier normally a userid or icn but anything will work
    * @param message the actual message
    * @param messageType the message type or function under which this is logged
    */
   @Transactional(propagation=Propagation.REQUIRES_NEW)
   public void logOutboundMessage(String userIdentifier, String messageType, String message) {

      logMessage(userIdentifier, messageType, message, "out");
   }

   /**
    * Writes a log entry out to the tracking table to assist in problem resolution. NOTE: This is
    * more of a debug-until-stable tool and is skewed toward HL7 messages but will work for other
    * types.
    */
   private void logMessage(String patientId, String messageType, String message, String direction) {

		Date now = new Date();

		// create prescription request
		MessageTraceLog messageTraceLog = new MessageTraceLog();

		messageTraceLog.setMessageType(messageType);
		messageTraceLog.setCreatedDate(now);
		messageTraceLog.setMessage(message);
		messageTraceLog.setMessageDirection(direction);
		messageTraceLog.setPatientId(patientId);

		try{
			//if (logger != null) {
				//logger.debug(">>>>>>>>>>>> before saving  messageTraceLog::"+messageTraceLog);
			//}
			messageTraceLog = messageTraceLogRepository.save(messageTraceLog);
		} catch(Exception e) {
			//logger.error("Error in storing messageTraceLog ", e);
			throw new MHVRuntimeException("Error in storing messageTraceLog " + e.getMessage(), e);
		}
   }
}
